import acrort
import os.path

from . cmdline_parser import ArgumentsProcessor, ArgumentsHelp, ArgumentParser
from . viewspec import DmlViewSpec


TAG_REQUEST = 'dml.request'
TAG_WORKFLOW = 'workflow'
TAG_PATTERN = 'pattern'
TAG_OPTIONS = 'options'
TAG_SYNCSITE = 'syncsite'

TAG_VALUE_REPORT = 'report'

# report is included as initial part of subscription
TAG_VALUE_SUBSCRIBE = 'subscribe'
# repost is skipped
TAG_VALUE_NEWS = 'news'
# report is feeded first before other news
TAG_VALUE_SUBSCRIBE_WITH_REPORT = 'subscribe-with-report'

WORKFLOW_CHOISE_SET = [TAG_VALUE_REPORT, TAG_VALUE_SUBSCRIBE, TAG_VALUE_NEWS, TAG_VALUE_SUBSCRIBE_WITH_REPORT]


class PostOffice:
    def __init__(self, connection, queue):
        self._source = connection.dml
        self._outbox = queue

    def report(self, spec):
        return self._source.report(spec, self._outbox)

    def subscribe(self, spec, report):
        return self._source.subscribe(spec, self._outbox, report=report)

    def execute_request(self, config):
        request = config[TAG_REQUEST]
        workflow = request[TAG_WORKFLOW]
        pattern = request[TAG_PATTERN]
        options = request.get(TAG_OPTIONS)

        spec = acrort.dml.ViewSpec(pattern, options)

        if workflow == TAG_VALUE_REPORT:
            return self.report(spec)

        elif workflow == TAG_VALUE_NEWS:
            return self.subscribe(spec, report=False)

        elif workflow == TAG_VALUE_SUBSCRIBE_WITH_REPORT:
            return self.subscribe(spec, report=True)

        elif workflow == TAG_VALUE_SUBSCRIBE:
            return self.subscribe(spec, report=None)

        acrort.common.make_logic_error("Unknown workflow falue '{}'.".format(workflow)).throw()


def describe_std_specs(spec_list):
    num = 0
    output = 'Numbers for standard DML View Specs are the following:\n'
    for spec in spec_list:
        num += 1
        output += "  {: >3} - {}\n".format(num, spec.description)
    return output


class PostOfficeArgumentsProcessor(ArgumentsProcessor):
    def __init__(self, std_specs=[]):
        ArgumentsProcessor.__init__(self)
        self._std_specs = std_specs

    def get_help(self):
        if not self._std_specs:
            return ArgumentsHelp()
        stdspecs_description = describe_std_specs(self._std_specs)
        return ArgumentsHelp(epilog=stdspecs_description)

    def create_arguments_parser(self):
        parser = ArgumentParser(add_help=False)
        parser.add_argument('-dw', '--dml-workflow', default=TAG_VALUE_REPORT, choices=WORKFLOW_CHOISE_SET)
        viewspec = parser.add_mutually_exclusive_group(required=True)
        viewspec.add_argument('-vs', '--viewspec-string', nargs=1, metavar='PATTERN')
        viewspec.add_argument('-vf', '--viewspec-file', nargs=1, metavar='FILE')
        if self._std_specs:
            viewspec.add_argument('--viewspec-std', nargs=1, type=int, metavar='NUMBER')
        parser.add_argument('-sm', '--source-machine', metavar='GUID')
        return parser

    def process_arguments(self, config, args):
        viewspec = None
        if args.viewspec_string is not None:
            pattern = args.viewspec_string[0]
            viewspec = DmlViewSpec.from_string(pattern)
            if args.source_machine is not None:
                viewspec.apply_source_machine(args.source_machine)
        elif args.viewspec_file is not None:
            filename = args.viewspec_file[0]
            viewspec = DmlViewSpec.load_from_file(filename)
        elif self._std_specs and args.viewspec_std is not None:
            num = args.viewspec_std[0]
            if num < 1 or num > len(self._std_specs):
                acrort.common.make_logic_error("Number '{}' is out of range for standard DML View Specs.".format(num)).throw()
            viewspec = self._std_specs[num - 1]
        if viewspec is None:
            acrort.common.make_logic_error("DML View Spec is required.").throw()
        plainspec = viewspec.request

        output = {}
        output[TAG_WORKFLOW] = args.dml_workflow
        output[TAG_PATTERN] = plainspec[0]
        output[TAG_OPTIONS] = plainspec[1]

        config[TAG_REQUEST] = output
