# @copyright (c) 2002-2013 Acronis International GmbH. All rights reserved.

from xml.etree import ElementTree
import logging
import sys

_VERSION = '2.0'
_XML_CONTENT_TYPE = 'text/xml'


class Error(Exception):
    def __init__(self, code, description):
        self.code = code
        self.description = description


def _parse_recursive(element):
    text = element.text and element.text.strip()
    kind = element.get('kind', None)
    if text:
        if kind == 'Int':
            return int(text)
        if kind == 'Float':
            return float(text)
        return text

    if kind == 'List':
        return [{child.tag: _parse_recursive(child)} for child in element]

    if not(len(element)):
        return ''

    result = {}
    for child in element:
        subelement = _parse_recursive(child)
        if isinstance(result, list):
            result.append({child.tag: subelement})
        elif child.tag in result:
            result = [result, {child.tag: subelement}]
        else:
            result[child.tag] = subelement
    return result


def _unparse_recursive(data, parent):
    for k, v in data.items():
        el = ElementTree.SubElement(parent, k)
        if isinstance(v, dict):
            _unparse_recursive(v, el)
        elif isinstance(v, (list, tuple)):
            el.set('kind', 'List')
            for node in v:
                _unparse_recursive(node, el)
        elif isinstance(v, int):
            el.set('kind', 'Int')
            el.text = str(v)
        elif isinstance(v, float):
            el.set('kind', 'Float')
            el.text = str(v)
        else:
            el.text = str(v)


def loads(text):
    return text and _parse_recursive(ElementTree.fromstring(text)) or {}


def dumps(data):
    root = ElementTree.Element('RequestRPC')
    root.attrib['version'] = _VERSION
    _unparse_recursive(data, root)
    return ElementTree.tostring(root, 'unicode')


def write(request, response):
    request.setStatus(200)
    request.setContentType(_XML_CONTENT_TYPE)
    request.write(dumps(response).encode('utf8'))


def __send_error(request, e, code, description):
    description = description or 'Unknown Error'
    logging.exception('XML-RPC. request failed: {0}', e, status=code, reason=description)
    if code == 401:  # unauthorised
        write(request, {'Status': 'BadCredentials', 'Description': description})
    elif code == 403:  # forbidden
        write(request, {'Status': 'BadCredentials', 'Description': 'Access Denied'})
    elif code == 423:  # locked
        write(request, {'Status': 'BrandAccountDisabled', 'Description': 'User is disabled'})
    elif code < 500 and code != 449:
        write(request, {'Status': 'Error', 'Description': description})
    else:
        # in this case client should retry request
        request.sendError(code, description)


def handle_error(request, error=None):
    try:
        raise error or sys.exc_info()[1]
    except Error as e:
        __send_error(request, e, e.code, e.description)
    except Exception as e:
        __send_error(request, e, 500, str(e))
