import acrort


def get_trait_value(trait_name, object):
    for name, trait_value in object.traits:
        if trait_name == name:
            return trait_value.ref
    return None


def create_viewspec_by_pattern(pattern):
    """Creates DML viewspec for specific objects by pattern."""
    return acrort.dml.ViewSpec(acrort.plain.Unit(flat=pattern))


def create_viewspec_by_is(object_is_trait):
    """Creates DML viewspec for specific objects type."""
    pattern = [('^Is', 'string', object_is_trait)]
    return acrort.dml.ViewSpec(acrort.plain.Unit(flat=pattern))


def create_viewspec_by_is_and_guid_property(object_is_trait, path_to_guid, guid):
    """Creates DML viewspec for specific object by type and guid property."""
    pattern = [
        ('^Is', 'string', object_is_trait),
        (path_to_guid, 'guid', guid),
    ]
    return acrort.dml.ViewSpec(acrort.plain.Unit(flat=pattern))


def create_viewspec_by_is_and_string_property(object_is_trait, path_to_string, str):
    """Creates DML viewspec for specific object by type and string property."""
    pattern = [
        ('^Is', 'string', object_is_trait),
        (path_to_string, 'string', str),
    ]
    return acrort.dml.ViewSpec(acrort.plain.Unit(flat=pattern))


def create_viewspec_by_is_and_id(object_is_trait, id):
    """Creates DML viewspec for specific object by type and id."""
    return create_viewspec_by_is_and_guid_property(object_is_trait, '.ID', id)


def create_viewspec_count(object_is_trait):
    """Creates DML viewspec for count."""
    actPattern = [('^Is', 'string', object_is_trait)]
    countOption = [('.Counter.CounterObjectTemplate.ID', 'string', 'Count'), ('.Counter.CounterObjectTemplate.ID^PrimaryKey', 'nil', None)]
    return acrort.dml.ViewSpec(acrort.plain.Unit(flat=actPattern), acrort.plain.Unit(flat=countOption))


def count_objects(connection, object_is_trait):
    """Counts DML objects with corresponding 'Is' trait."""
    spec = create_viewspec_count(object_is_trait)
    count = connection.dml.select1(spec).CounterValue.ref
    return count


def count_objects_by_spec(connection, spec):
    """Counts DML objects matching by specified spec."""
    countOption = [('.Counter.CounterObjectTemplate.ID', 'string', 'Count'), ('.Counter.CounterObjectTemplate.ID^PrimaryKey', 'nil', None)]
    countOptionUnit = acrort.plain.Unit(flat=countOption)
    spec_options = spec.options.consolidate(countOptionUnit) if spec.options else countOptionUnit
    count = connection.dml.select1(acrort.dml.ViewSpec(spec.pattern, spec_options)).CounterValue.ref
    return count


def create_viewspec_by_is_and_like(is_trait, property_path, value):
    """Creates DML viewspec for objects matching specified property using Like."""
    pattern = [
        ('^Is', 'string', is_trait),
        (property_path, 'string', ''),
        ('{0}^Like'.format(property_path), 'string', '%{0}%'.format(value)),
    ]
    return acrort.dml.ViewSpec(pattern=acrort.plain.Unit(flat=pattern))


def create_viewspec_tenant_by_name(tenant_name):
    """Creates DML viewspec for 'Tenants::HierarchyNode' objects matching specified name."""
    return create_viewspec_by_is_and_like('Tenants::HierarchyNode', '.Name', tenant_name)


def create_viewspec_tenant(tenant_id):
    """Creates DML viewspec for specific 'Tenants::HierarchyNode' object."""
    pattern = [
        ('^Is', 'string', 'Tenants::HierarchyNode'),
        ('.ID', 'string', tenant_id),
    ]
    return acrort.dml.ViewSpec(pattern=acrort.plain.Unit(flat=pattern))


def create_viewspec_tenants():
    """Creates DML viewspec for all 'Tenants::HierarchyNode' objects."""
    pattern = [
        ('^Is', 'string', 'Tenants::HierarchyNode'),
    ]
    return acrort.dml.ViewSpec(pattern=acrort.plain.Unit(flat=pattern))


def create_viewspec_backup_accounts():
    """Creates DML viewspec for all 'Tenants::HierarchyNode' objects."""
    pattern = [
        ('^Is', 'string', 'Tenants::HierarchyNode'),
        ('.Kind', 'sdword', -1),
    ]
    return acrort.dml.ViewSpec(pattern=acrort.plain.Unit(flat=pattern))


def create_viewspec_instances_by_tenant_id(tenant_id):
    """Creates DML pattern for all 'InstanceManagement::Instance' objects of the specified tenant."""
    pattern = [
        ('^Is', 'string', 'InstanceManagement::Instance'),
        ('.Tenant.ID', 'string', tenant_id),
    ]
    return acrort.dml.ViewSpec(pattern=acrort.plain.Unit(flat=pattern))


def create_viewspec_machines_by_tenant_id(tenant_id):
    """Creates DML pattern for all 'MachineManagement::Machine' objects of the specified tenant."""
    pattern = [
        ('^Is', 'string', 'MachineManagement::Machine'),
        ('.Tenant.ID', 'string', tenant_id),
    ]
    return acrort.dml.ViewSpec(pattern=acrort.plain.Unit(flat=pattern))


def create_viewspec_machines_by_role(role):
    """Creates DML pattern for all 'MachineManagement::Machine' objects with the specified role."""
    pattern = [
        ('^Is', 'string', 'MachineManagement::Machine'),
        ('.Info.Role', 'dword', role),
    ]
    return acrort.dml.ViewSpec(pattern=acrort.plain.Unit(flat=pattern))


def create_pattern_by_timestamp_greater(timestamp):
    """Creates DML pattern for all objects with DmlTimeStmap greater than timestamp."""
    if timestamp == 0:
        return acrort.plain.Unit(flat=[('.DmlTimeStamp', 'sqword', 0), ('.DmlTimeStamp^GreaterEqual', 'sqword', timestamp)])
    return acrort.plain.Unit(flat=[('.DmlTimeStamp', 'sqword', 0), ('.DmlTimeStamp^Greater', 'sqword', timestamp)])


def create_pattern_by_timestamp_less(timestamp):
    """Creates DML pattern for all objects with DmlTimeStmap less than timestamp."""
    return acrort.plain.Unit(flat=[('.DmlTimeStamp', 'sqword', 0), ('.DmlTimeStamp^Less', 'sqword', timestamp)])


def create_viewspec_msp_machine_by_owner_id(owner_id):
    """Creates DML viewspec for all 'Msp::AMS::Dto::Machine' objects by owner id."""
    pattern = [
        ('^Is', 'string', 'Msp::AMS::Dto::Machine'),
        ('.OwnerID', 'string', owner_id),
    ]
    return acrort.dml.ViewSpec(pattern=acrort.plain.Unit(flat=pattern))


def create_viewspec_msp_machine_by_sequenced_id(sequenced_id):
    """Creates DML viewspec for all 'Msp::AMS::Dto::Machine' objects by sequenced id."""
    pattern = [
        ('^Is', 'string', 'Msp::AMS::Dto::Machine'),
        ('.SequencedID', 'qword', int(sequenced_id)),
    ]
    return acrort.dml.ViewSpec(pattern=acrort.plain.Unit(flat=pattern))


def create_viewspec_machine_by_id(machine_id):
    """Creates DML viewspec for specific 'MachineManagement::Machine' with specified id."""
    pattern = [
        ('^Is', 'string', 'MachineManagement::Machine'),
        ('.ID', 'guid', machine_id),
    ]
    return acrort.dml.ViewSpec(pattern=acrort.plain.Unit(flat=pattern))


def create_viewspec_deployed_protection_plan_by_id(plan_id):
    """Creates DML viewspec for deployed 'Gtob::Dto::ProtectionPlan' with specified id."""
    pattern = [
        ('^Is', 'string', 'Gtob::Dto::ProtectionPlan'),
        ('.Origin', 'dword', 2),
        ('.ID', 'guid', plan_id),
    ]
    return acrort.dml.ViewSpec(pattern=acrort.plain.Unit(flat=pattern))


def create_viewspec_instance_by_machine_and_type(machine_id, type):
    """Creates DML viewspec for specific 'InstanceManagement::Instance' with specified host id and type."""
    pattern = [
        ('^Is', 'string', 'InstanceManagement::Instance'),
        ('.HostID', 'guid', machine_id),
        ('.Type', 'dword', type),
    ]
    return acrort.dml.ViewSpec(pattern=acrort.plain.Unit(flat=pattern))


def viewspec_apply_mask(spec, mask):
    """Applies mask for DML viewspec."""
    spec_options = spec.options.consolidate(mask) if spec.options else mask
    return acrort.dml.ViewSpec(spec.pattern, spec_options)


def viewspec_apply_tenants(spec, tenants_ids):
    """Applies filtering by tenants for DML viewspec."""
    if not tenants_ids:
        return spec

    ids_pattern = []
    for tenant_id in tenants_ids:
        ids_pattern.append([('', 'string', tenant_id)])

    pattern = [
        ('.Tenant.ID', 'string', ''),
        ('.Tenant.ID^ValueIn', 'complex_trait', [('', 'array', ids_pattern)]),
    ]

    return acrort.dml.ViewSpec(spec.pattern.consolidate(acrort.plain.Unit(flat=pattern)), spec.options)


def viewspec_apply_tenant_locator(spec, locator):
    """Applies filtering by tenant locator for DML viewspec."""
    if not locator:
        return spec

    pattern = [
        ('.Tenant.Locator', 'string', ''),
        ('.Tenant.Locator^DirectRelative', 'string', locator)
    ]

    return acrort.dml.ViewSpec(spec.pattern.consolidate(acrort.plain.Unit(flat=pattern)), spec.options)


def viewspec_apply_ids(spec, ids):
    """Applies filtering by ids for DML viewspec."""
    if not ids:
        return spec

    ids_pattern = []
    for id in ids:
        ids_pattern.append([('', 'guid', id)])

    pattern = [
        ('.ID', 'guid', '00000000-0000-0000-0000-000000000000'),
        ('.ID^ValueIn', 'complex_trait', [('', 'array', ids_pattern)]),
    ]

    return acrort.dml.ViewSpec(spec.pattern.consolidate(acrort.plain.Unit(flat=pattern)), spec.options)


def viewspec_apply_timestamp_less(spec, timestamp):
    """Applies filtering by timestamp for DML viewspec."""
    if not timestamp:
        return spec

    timestamp_pattern = create_pattern_by_timestamp_less(timestamp)
    spec_pattern = spec.pattern.consolidate(timestamp_pattern)

    return acrort.dml.ViewSpec(spec_pattern, spec.options)


def viewspec_apply_remote_host(spec, host_id):
    """Applies __source_machine tag for specified DML viewspec."""
    pattern = [
        ('.__source_machine', 'guid', host_id),
    ]

    return acrort.dml.ViewSpec(spec.pattern.consolidate(acrort.plain.Unit(flat=pattern)), spec.options)


def viewspec_apply_source(spec, host_id):
    """Applies ^Source trait for specified DML viewspec."""
    pattern = [
        ('^Source', 'string', str(host_id))
    ]

    return acrort.dml.ViewSpec(spec.pattern.consolidate(acrort.plain.Unit(flat=pattern)), spec.options)


def create_mask(property_path):
    """Creates DML options mask by property name."""
    return acrort.plain.Unit(flat=[
        ('.Mask{0}'.format(property_path), 'nil', None)])


def create_mask2(property_path1, property_path2):
    """Creates DML options mask by properties names."""
    return acrort.plain.Unit(flat=[
        ('.Mask{0}'.format(property_path1), 'nil', None),
        ('.Mask{0}'.format(property_path2), 'nil', None)])


def create_mask3(property_path1, property_path2, property_path3):
    """Creates DML options mask by properties names."""
    return acrort.plain.Unit(flat=[
        ('.Mask{0}'.format(property_path1), 'nil', None),
        ('.Mask{0}'.format(property_path2), 'nil', None),
        ('.Mask{0}'.format(property_path3), 'nil', None)])


def create_mask4(property_path1, property_path2, property_path3, property_path4):
    """Creates DML options mask by properties names."""
    return acrort.plain.Unit(flat=[
        ('.Mask{0}'.format(property_path1), 'nil', None),
        ('.Mask{0}'.format(property_path2), 'nil', None),
        ('.Mask{0}'.format(property_path3), 'nil', None),
        ('.Mask{0}'.format(property_path4), 'nil', None)])


def create_options_sorted_by_timestamp(limit):
    """Creates DML options sorted by DmlTimeStamp with limit."""
    return acrort.plain.Unit(flat=[
        ('.LimitOptions', 'dword', limit),
        ('.SortingOptions.DmlTimeStamp', 'sqword', 0)])


def select_objects(connection, spec):
    """Selects all DML objects using spec."""
    objects = connection.dml.select(spec)
    return objects


def select_object(connection, spec):
    """Selects first DML object matching the spec."""
    objects = select_objects(connection, spec)
    if len(objects) > 0:
        return objects[0]
    return None


def safe_select_objects(connection, spec):
    """Selects all DML objects matching the spec and returns select operation result (True/False, objects)."""
    objects = None
    try:
        objects = select_objects(connection, spec)
    except acrort.Exception as _:
        return (False, objects)

    return (True, objects)


def safe_select_object(connection, spec):
    """Selects first DML object matching the spec and returns select operation result (True/False, object)."""
    objects = None
    try:
        objects = select_objects(connection, spec)
    except acrort.Exception as _:
        return (False, objects)
    if objects is not None and len(objects) > 0:
        return (True, objects[0])
    return (False, None)


def get_max_timestamp_for_objects(connection, spec):
    options = [
        ('.Mask.DmlTimeStamp', 'nil', None),
        ('.SortingOptions.DmlTimeStamp', 'sqword', 0),
        ('.SortingOptions.DmlTimeStamp^Descending', 'nil', None),
        ('.LimitOptions', 'dword', 1),
    ]
    options_unit = acrort.plain.Unit(flat=options)
    spec_options = spec.options.consolidate(options_unit) if spec.options else options_unit
    (selected, object) = safe_select_object(connection, acrort.dml.ViewSpec(spec.pattern, spec_options))
    if selected:
        return object.DmlTimeStamp.ref
    return None


def enum_objects(connection, spec, chunk_size=100):
    """
        Generator for enumerating all DML objects according to viewspec.
        Example:
            c = acrort.connectivity.Connection('ams')
            pattern = [ ('^Is', 'string', 'Tol::History::Plain::Activity') ]
            for activity in dml_utils.enum_objects(c, spec):
                print (activity.ID.ref)
    """

    q = acrort.common.Queue(sigint=True)
    request_next = True
    dml_timestamp = 0

    timestamp_pattern = create_pattern_by_timestamp_greater(dml_timestamp)
    spec_pattern = spec.pattern.consolidate(timestamp_pattern)
    spec_options = spec.options.consolidate(create_options_sorted_by_timestamp(chunk_size)) if spec.options else create_options_sorted_by_timestamp(chunk_size)
    if 'Mask' in spec_options:
        spec_options = spec_options.consolidate(create_mask('.DmlTimeStamp'))
    connection.dml.report(acrort.dml.ViewSpec(spec_pattern, spec_options), q)

    expected_messages = (acrort.common.MESSAGE_TYPE_COMPLETION, acrort.common.MESSAGE_TYPE_DML_REPORT_COMPLETION, acrort.common.MESSAGE_TYPE_DML_REPORT)

    while True:
        message = q.get()

        if message.type_id not in expected_messages:
            raise RuntimeError("Interrupt on message '{0}'".format(message))

        if message.type_id == acrort.common.MESSAGE_TYPE_DML_REPORT_COMPLETION:
            continue

        if message.type_id == acrort.common.MESSAGE_TYPE_COMPLETION:
            if request_next is False:
                break
            # query next part
            request_next = False
            timestamp_pattern = create_pattern_by_timestamp_greater(dml_timestamp)
            spec_pattern = spec.pattern.consolidate(timestamp_pattern)
            sort_options = create_options_sorted_by_timestamp(chunk_size)
            spec_options = spec.options.consolidate(sort_options) if spec.options else sort_options
            if 'Mask' in spec_options:
                spec_options = spec_options.consolidate(create_mask('.DmlTimeStamp'))
            connection.dml.report(acrort.dml.ViewSpec(spec_pattern, spec_options), q)
            continue

        yield message.object
        request_next = True
        dml_timestamp = message.object.DmlTimeStamp.ref


def enumerate_objects(connection, spec, callback, error_callback, chunk_size=100):
    """Enumerates all DML objects according to viewspec."""

    try:
        for object in enum_objects(connection, spec, chunk_size):
            callback(object)
    except Exception as e:
        error_callback("Interrupt on message '{0}'".format(e))


def enumerate_objects_by_is(connection, is_trait, callback, error_callback):
    """Enumerates all DML objects according to 'Is' tarit."""
    enumerate_objects(connection, create_viewspec_by_is(is_trait), callback, error_callback)


def enumerate_tenants(connection, callback, error_callback):
    """Enumerates all 'Tenants::HierarchyNode' DML objects."""
    enumerate_objects(connection, create_viewspec_tenants(), callback, error_callback)


def enumerate_backup_accounts(connection, callback, error_callback):
    """Enumerates all 'Tenants::HierarchyNode' DML objects with 'Kind' == -1."""
    enumerate_objects(connection, create_viewspec_backup_accounts(), callback, error_callback)
