import os
import glob
import zipfile
import acrort

from . os_utils import load_py_object


TAG_VIEWSPEC_PATTERN = 'pattern'
TAG_VIEWSPEC_OPTIONS = 'options'
TAG_VIEWSPEC_DESCRIPTION = 'description'
SPEC_NODE_PATTERN = '.spec.txt'


class SpecStorageBase:
    def enumerate_nodes(self):
        return []

    def enumerate(self):
        nodes = self.enumerate_nodes()
        result = []
        for node in nodes:
            spec = self.load_spec(node)
            result.append(spec)
        result.sort(key=lambda spec: spec.description)
        return result

    def load_object(self, node):
        acrort.common.make_logic_error("ViewSpec Storage: node '{}' not found.".format(node)).throw()

    def unit_from_source(self, flat_or_node):
        if isinstance(flat_or_node, list):
            return acrort.plain.Unit(flat=flat_or_node)
        source = self.load_object(flat_or_node)
        return acrort.plain.Unit(flat=source)

    def describe_node(self, node):
        return node

    def load_spec(self, node):
        default_description = self.describe_node(node)
        source = self.load_object(node)
        if isinstance(source, list):
            pattern = acrort.plain.Unit(flat=source)
            return DmlViewSpec(pattern=pattern, description=default_description)
        elif type(source) == tuple:
            sz = len(source)
            if sz == 1:
                pattern = self.unit_from_source(source[0])
                return DmlViewSpec(pattern=pattern, description=default_description)
            elif sz == 2:
                pattern = self.unit_from_source(source[0])
                options = self.unit_from_source(source[1])
                return DmlViewSpec(pattern=pattern, options=options, description=default_description)
            else:
                acrort.common.make_logic_error("Malformed DML ViewSpec.").throw()
        elif type(source) == dict:
            pattern = self.unit_from_source(source[TAG_VIEWSPEC_PATTERN])
            options = None
            options_ref = source.get(TAG_VIEWSPEC_OPTIONS)
            if options_ref is not None:
                options = self.unit_from_source(options_ref)
            description = source.get(TAG_VIEWSPEC_DESCRIPTION, default_description)
            if options is None:
                return DmlViewSpec(pattern=pattern, description=description)
            return DmlViewSpec(pattern=pattern, options=options, description=description)
        else:
            acrort.common.make_logic_error("Malformed DML ViewSpec.").throw()


class DirSpecStorage(SpecStorageBase):
    def __init__(self, location, pattern=SPEC_NODE_PATTERN):
        glob = ''.join(['*', pattern])
        self.location = location
        self.glob_pattern = os.path.join(location, glob)

    def describe_node(self, node):
        name = os.path.basename(node)
        if name.endswith(SPEC_NODE_PATTERN):
            name = name[0:len(name) - len(SPEC_NODE_PATTERN)]
        return name

    def load_object(self, node):
        path = os.path.normpath(node)
        if not os.path.isabs(path):
            path = os.path.join(self.location, path)
        return load_py_object(path)

    def enumerate_nodes(self):
        return glob.glob(self.glob_pattern)


class SingleFileSpecStorage(SpecStorageBase):
    def __init__(self, filename):
        path = os.path.normpath(filename)
        if not os.path.isabs(path):
            path = os.path.abspath(path)
        self.filepath = path
        self.location = os.path.dirname(path)

    def describe_node(self, node):
        return os.path.basename(node)

    def load_object(self, node):
        path = os.path.normpath(node)
        if not os.path.isabs(path):
            path = os.path.join(self.location, path)
        return load_py_object(path)

    def enumerate_nodes(self):
        return [self.filepath]


class ZipSpecStorage(SpecStorageBase):
    def __init__(self, zip_filename, pattern=SPEC_NODE_PATTERN):
        self.zipobject = zipfile.ZipFile(zip_filename, 'r')
        self.catalog = self.zipobject.namelist()
        self.pattern = pattern

    def load_object(self, node):
        if node not in self.catalog:
            acrort.common.make_logic_error("ViewSpec Storage: node '{}' not found.".format(node)).throw()
        bytes = self.zipobject.open(node).read()
        ast = compile(bytes, node, 'eval')
        return eval(ast, {"__builtins__": None}, {})

    def enumerate_nodes(self):
        nodes = []
        for item in self.catalog:
            if item.endswith(self.pattern):
                nodes.append(item)
        return nodes


def create_viewspec_storage(name, location):
    store_name = os.path.join(location, name)
    if not os.path.exists(store_name):
        zip_name = ''.join([store_name, '.zip'])
        if os.path.exists(zip_name):
            store_name = zip_name
    if os.path.exists(store_name) and os.path.isdir(store_name):
        return DirSpecStorage(store_name)
    if os.path.exists(store_name) and zipfile.is_zipfile(store_name):
        return ZipSpecStorage(store_name)
    return SpecStorageBase()


class DmlViewSpec:
    def __init__(self, pattern, options=None, description=''):
        self._pattern = pattern
        self._options = options
        self._description = description

    @property
    def request(self):
        return (self._pattern, self._options)

    @property
    def description(self):
        return self._description

    @staticmethod
    def from_string(subject):
        flat = [('', 'empty_composite', None), ('^Is', 'string', subject)]
        pattern = acrort.plain.Unit(flat=flat)
        return DmlViewSpec(pattern=pattern, description=subject)

    @staticmethod
    def load_from_file(filename):
        try:
            store = SingleFileSpecStorage(filename)
            specs = store.enumerate()
            return specs[0]
        except acrort.Exception as exception:
            details = exception.to_error()
            error = acrort.common.make_logic_error_with_suberror("Failed to load DML ViewSpec from file: '{}'".format(filename), details)
            error.throw()

    def apply_source_machine(self, machine_id):
        flat = [
            ('.__source_machine', 'guid', machine_id),
        ]
        self._pattern = self._pattern.consolidate(acrort.plain.Unit(flat=flat))
