from . constants import *
from . slices import *
import datetime
import acrobind.protection
import acrort

def week_number(date, week_beginning=1):
    return (date - datetime.timedelta(days=week_beginning - 1)).isocalendar()[1]


class SliceInFutureError(Exception):
    def __init__(self, slice):
        self.slice = slice


def SliceInFutureErrorToLogicError(error, stage):
    return acrort.common.make_logic_error("Backup with creation date in future was detected. Path to archive: '{}', archive name: '{}', backup ID: '{}', backup creation date and time: '{}'. Please, delete this backup and start backup again.".
        format(stage.LocationUri.ref, stage.ArchiveName.ref, error.slice['id'], error.slice['creation_time']))


def calculate_simple_backup_set(slice, predecessor, schedule_type, week_beginning=1, actual_sets = [BACKUP_SET_MONTHLY, BACKUP_SET_WEEKLY, BACKUP_SET_DAILY, BACKUP_SET_HOURLY]):
    backup_set_checks = {
        BACKUP_SET_MONTHLY: (lambda schedule_type, slice_time, predecessor_time : True if schedule_type == 'monthly' or slice_time.year != predecessor_time.year or slice_time.month != predecessor_time.month else False),
        BACKUP_SET_WEEKLY: (lambda schedule_type, slice_time, predecessor_time : True if schedule_type == 'weekly' or slice_time.day != predecessor_time.day and week_number(slice_time, week_beginning) != week_number(predecessor_time, week_beginning) else False),
        BACKUP_SET_DAILY: (lambda schedule_type, slice_time, predecessor_time : True if schedule_type == 'daily' or predecessor_time.day != slice_time.day else False),
        BACKUP_SET_HOURLY: (lambda schedule_type, slice_time, predecessor_time : True),
    }
    if len(actual_sets) == 0:
        acrort.common.make_logic_error('actual_sets list must not be empty').throw()
    if predecessor is None:
        return actual_sets[0]
    slice_time = slice['creation_time']
    predecessor_time = predecessor['creation_time']
    if slice_time <= predecessor_time:
        raise SliceInFutureError(predecessor)
    for backup_set in actual_sets:
        if backup_set_checks[backup_set](schedule_type, slice_time, predecessor_time):
            return backup_set
    return actual_sets[len(actual_sets)-1]


def calculate_weekly_daily_backup_set(slice, predecessor, week_beginning=1):
    return calculate_simple_backup_set(slice, predecessor, 'daily', week_beginning, [BACKUP_SET_WEEKLY, BACKUP_SET_DAILY])


def calculate_monthly_weekly_daily_backup_set(slice, predecessor, week_beginning=1):
    return calculate_simple_backup_set(slice, predecessor, 'daily', week_beginning, [BACKUP_SET_MONTHLY, BACKUP_SET_WEEKLY, BACKUP_SET_DAILY])


def get_schedule_type(backup_plan, stage_index = 0):
    schedule_types = {
        SCHEDULE_TYPE_NOT_SPECIFIED: None,
        SCHEDULE_TYPE_HOURLY: 'hourly',
        SCHEDULE_TYPE_DAILY: 'daily',
        SCHEDULE_TYPE_WEEKLY: 'weekly',
        SCHEDULE_TYPE_MONTHLY: 'monthly'
    }
    if backup_plan.Scheme.Type.ref == acrobind.protection.BACKUP_SCHEME_SIMPLE and 'Type' not in backup_plan.Scheme.Parameters.BackupSchedule.Schedule:
        return 'old-style-simple'
    single_schedule_type = lambda: schedule_types[backup_plan.Scheme.Parameters.BackupSchedule.Schedule.Type.ref]
    scheme_schedule_types = {
        acrobind.protection.BACKUP_SCHEME_CUSTOM: (lambda: 'custom'),
        acrobind.protection.BACKUP_SCHEME_WEEKLY_FULL_DAILY_INCR: (lambda: 'weekly-daily'),
        acrobind.protection.BACKUP_SCHEME_MONTHLY_FULL_WEEKLY_DIFF_DAILY_INCR: (lambda: 'monthly-weekly-daily'),
        acrobind.protection.BACKUP_SCHEME_ALWAYS_FULL: single_schedule_type,
        acrobind.protection.BACKUP_SCHEME_ALWAYS_INCREMENTAL: single_schedule_type,
        acrobind.protection.BACKUP_SCHEME_SIMPLE: single_schedule_type,
        acrobind.protection.BACKUP_SCHEME_GFS: (lambda: 'gfs')
    }
    get_schedule_type = scheme_schedule_types[backup_plan.Scheme.Type.ref]
    schedule_type = get_schedule_type()

    if schedule_type == 'weekly' or schedule_type == 'monthly':
        stages = [make_stage_spec(backup_plan, stage) for _, stage in backup_plan.Route.Stages]
        stage_count = len(stages)
        if stage_index >= stage_count:
            acrort.common.make_logic_error('stage_index={} while stage_count={}'.format(stage_index, stage_count)).throw()

        source_stage = stages[stage_index]
        for _, item in source_stage['rules']:
            if item.BackupSetIndex.ref == BACKUP_SET_DAILY:
                schedule_type = 'daily'
                break
    
    return schedule_type

def create_backup_set_calculator(schedule_type, week_beginning=1):
    slice_kind_mapping = {
        SLICE_KIND_FULL: BACKUP_SET_FULL,
        SLICE_KIND_DIFFERENTIAL: BACKUP_SET_DIFF,
        SLICE_KIND_INCREMENTAL: BACKUP_SET_INC,
        SLICE_KIND_NA: BACKUP_SET_FULL,
        SLICE_KIND_CDP: BACKUP_SET_INC,
    }
    calculate_by_kind = lambda slice, predecessor: slice_kind_mapping[slice['kind']]
    calculators = {
        'monthly': (lambda slice, predecessor: calculate_simple_backup_set(slice, predecessor, 'monthly', week_beginning)),
        'weekly': (lambda slice, predecessor: calculate_simple_backup_set(slice, predecessor, 'weekly', week_beginning)),
        'daily': (lambda slice, predecessor: calculate_simple_backup_set(slice, predecessor, 'daily', week_beginning)),
        'hourly': (lambda slice, predecessor: calculate_simple_backup_set(slice, predecessor, 'hourly', week_beginning)),
        'weekly-daily': (lambda slice, predecessor: calculate_weekly_daily_backup_set(slice, predecessor, week_beginning)),
        'monthly-weekly-daily': (lambda slice, predecessor: calculate_monthly_weekly_daily_backup_set(slice, predecessor, week_beginning)),
        'custom': calculate_by_kind,
        'gfs': calculate_by_kind,
        'old-style-simple': calculate_by_kind
    }
    if schedule_type in calculators:
        return calculators[schedule_type]
    
    return calculate_by_kind


def get_week_beginning(backup_plan):
    return 1 + (backup_plan.Options.BackupOptions.Additional.WeeklyBackupDay.ref + 6) % 7
